/* brushlib - The MyPaint Brush Library (demonstration project)
 * Copyright (C) 2013 POINTCARRE SARL / Sebastien Leon email: sleon at pointcarre.com
 *
 * Permission to use, copy, modify, and/or distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 */

#include <QColorDialog>

#include "MypaintView.h"
#include "Tile.h"
#include <assert.h>

//-------------------------------------------------------------------------
static MypaintView* s_view = NULL; // ugly turn-around to pass the view to the call-back, because there is no void* context in MyPaintTiledSurface

void CallBack_MyPaintTiledSurfaceTileRequestStartFunction (MyPaintTiledSurface *self, MyPaintTiledSurfaceTileRequestData *rq)
{
  // I guess that rq is only valid for "tx", "ty" and this function 
  // should fill the "buffer" (and eventually "context")
  Tile* tile = s_view->GetTileFromIdx(QPoint(rq->tx,rq->ty));
  assert(tile);
  rq->buffer = tile ? tile->Bits() : NULL;
  // (regarding the "context", this field is exploitable here as we can fill
  // it with the Tile pointer. It will save a call to "s_view->GetTileFromIdx"
  // Unfortunately, this can only be done primarly with the help of some ugly 
  // static (s_view here, because when we enter into this function there is no 
  // other way to get the view object pointer)
  // It would be very useful too, to get the same kind of "context" var
  // from the surface which can be initialized correctly by the owner object)
  rq->context = (void*)tile;
}

void CallBack_MyPaintTiledSurfaceTileRequestEndFunction (MyPaintTiledSurface *self, MyPaintTiledSurfaceTileRequestData *rq)
{
  Tile* tile = (Tile*)rq->context;
  if (tile)
  {
    assert(tile && tile == s_view->GetTileFromIdx(QPoint(rq->tx,rq->ty)));
    // If the Tile buffer has been modified, it is necessary to inform the Tile
    // so the cache display is updated when the tile is blitted to the screen
    if (!rq->readonly)
    {
      tile->InvalidCache ();
      tile->update();
    }
  }
}

void CallBack_MyPaintSurfaceBeginAtomic (MyPaintSurface *surf)
{
  // not useful in this simplistic demo (no undo or anything which is 
  // accessing to the tiles in parallel of the drawing code)
}

MyPaintRectangle* CallBack_MyPaintSurfaceEndAtomic (MyPaintSurface *surf)
{
  // What can I return ?? Make me think of MyPaintTiledSurfaceAreaChanged but
  // this callback (prototype still declared) is no longer called :-(
  // and here I have to return the rect (previously, it was the lib that returned the modified area...)
  return NULL;
}

void CallBack_MyPaintSurfaceDestroy (MyPaintSurface *surf)
{
  // There is nothing to do here as all tiles are managed by the Qt graphic view.
}

/*
void CallBack_MyPaintTiledSurfaceAreaChanged (MyPaintTiledSurface *self, int bb_x, int bb_y, int bb_w, int bb_h)
{
  // we already register an update for the touched tile...
}
*/
//-------------------------------------------------------------------------

void MypaintView::Init()
{
  assert(s_view == NULL);
  s_view = this;
  mp_brush = NULL;
  // clean it
  memset( m_tileTable, 0, sizeof(m_tileTable));
  // clean the TiledSurface
  memset( &m_tile, 0, sizeof(MyPaintTiledSurface));
  // Init tile surface. Any other init required ??
  mypaint_tiled_surface_init(
    &m_tile, 
    CallBack_MyPaintTiledSurfaceTileRequestStartFunction,
    CallBack_MyPaintTiledSurfaceTileRequestEndFunction);

  // no other way to fill these callback ? Seems ugly to go inside a private-like struct :
  m_tile.parent.begin_atomic = CallBack_MyPaintSurfaceBeginAtomic;
  m_tile.parent.end_atomic   = CallBack_MyPaintSurfaceEndAtomic;
  m_tile.parent.destroy      = CallBack_MyPaintSurfaceDestroy;
  // we do not care about saving to png
  // End of LibMyPaint Init...

  m_scene.setSceneRect ( QRect(QPoint(), QSize(k_tile_dim*k_max, k_tile_dim*k_max)) );
  setScene     (&m_scene);
  setAlignment (0);
  setVerticalScrollBarPolicy   (Qt::ScrollBarAlwaysOn);
  setHorizontalScrollBarPolicy (Qt::ScrollBarAlwaysOn);
  centerOn( GetTilePos(QPoint(k_center, k_center)) );
  setMouseTracking ( true ); // to get the mouse event even when the user does not click !
}

MypaintView::~MypaintView ()
{
  mypaint_brush_unref         (mp_brush); // Same that previous call to destroy ?
  // mypaint_surface_destroy has been removed from API. Is it handled automatically ?
  mypaint_tiled_surface_destroy(&m_tile);
}

#define CTILT(t) ((float)t/60) // CONVERT QT TILT to MYPAINT TILT

void MypaintView::tabletEvent(QTabletEvent *e)
{
  switch (e->type())
  {
    case QEvent::TabletPress:
      if (e->pointerType() == QTabletEvent::Pen)
      {
        GetDTime();
        e->accept();
      }
      break;
    case QEvent::TabletRelease:
      if (e->pointerType() == QTabletEvent::Pen)
      {
        e->accept();
      }
      break;
    case QEvent::TabletMove:
      if (e->pointerType() == QTabletEvent::Pen)
      {
        QPointF pt(mapToScene(e->pos()));
        MyPaintSurface* p_surf = &m_tile.parent;
        mypaint_surface_begin_atomic(p_surf);
        mypaint_brush_stroke_to(mp_brush, p_surf, pt.x(), pt.y(), e->pressure(), CTILT(e->xTilt()),  CTILT(e->yTilt()), GetDTime());
        mypaint_surface_end_atomic(p_surf);
        e->accept();
      }
      break;
   }
}

void MypaintView::mousePressEvent   ( QMouseEvent * e )
{
  GetDTime();
  m_mousePressed = (e->button() == Qt::LeftButton);
}
void MypaintView::mouseMoveEvent    ( QMouseEvent * e )
{
  QPointF pt(mapToScene(e->pos()));
  MyPaintSurface* p_surf = &m_tile.parent;
  mypaint_surface_begin_atomic(p_surf);
  // Note that we pass to libMyPaint all events, including the mouseMove when the
  // mouse button is NOT pressed. This is required to libMyPaint logic. (Martin's email / 2013/01/21)
  // But when the button is not pressed, the density is 0.
  // Surprisingly, in hover mode, for some brushes, we get the SurfaceTileRequest
  // (readonly of course, but the question is : is it normal ?)
  // When we are in hover mode, I am wondering if it would be better to simply store
  // the mouse move events (several event or just the last one or according the time tag)
  // and replay them (with correct time tags) at the very first click...
  mypaint_brush_stroke_to(mp_brush, p_surf, pt.x(), pt.y(), (m_mousePressed ? 1.0 : 0.0), 0, 0, GetDTime());
  mypaint_surface_end_atomic(p_surf);
}
void MypaintView::mouseReleaseEvent ( QMouseEvent * e )
{
  m_mousePressed = false;
}

void MypaintView::BrushSelected (const QByteArray& content)
{
  // user selected another brush. Time to reload it :
  if (mp_brush) mypaint_brush_unref(mp_brush); // Same than previous API Destroy ?
  mp_brush = mypaint_brush_new();
  bool loaded = mypaint_brush_from_string(mp_brush, content.constData());
  if (!loaded) qDebug("Trouble when reading the selected brush !"); // Not able to load the selected brush. !?
  // give the current color to the selected brush...
  UpdateBrushColor ();
  emit BrushChanged(mp_brush);
}

// Very basic Tile management (fixed number of maximum tiles)
Tile* MypaintView::GetTileFromIdx (const QPoint& idx)
{
  Tile* result = NULL;
  // Which tile index is it ?
  if (CheckIndex(idx.x()) && CheckIndex(idx.y())) // out of range ?
  {
    // Ok, valid index. Does it exist already ?
    result = m_tileTable [idx.x()][idx.y()];
    if (!result)
    {
      // Time to allocate it, update table and insert it as a QGraphicsItem in scene:
      result = new Tile();
      m_tileTable [idx.x()][idx.y()] = result;
      m_scene.addItem(result);
      QPoint tilePos ( GetTilePos(idx) );
      result->setPos(tilePos);
      result->show();
    }
  }
  return result;
}

void MypaintView::BtnChgColorPressed ()
{
  QPushButton* p_btn = dynamic_cast<QPushButton*>(sender());
  if (p_btn)
  {
    QColor newColor = QColorDialog::getColor ( m_color, window(), "Select the brush color", QColorDialog::ShowAlphaChannel );
    if (newColor.isValid())
    {
      p_btn->setStyleSheet( QString("color: %1; background-color: %2;").arg((newColor.lightnessF()>0.5)?"black":"white").arg(newColor.name()) );
      m_color = newColor;
      // Update libMyPaint with the new color...
      UpdateBrushColor ();
    }
  }
}

 void MypaintView::RadiusChanged  (int val)
 {
   float valF = (float)val / 100;
   mypaint_brush_set_base_value(mp_brush, MYPAINT_BRUSH_SETTING_RADIUS_LOGARITHMIC, valF);
 }

 void MypaintView::OpacityChanged (int val)
 {
   float valF = (float)val / 100;
   mypaint_brush_set_base_value(mp_brush, MYPAINT_BRUSH_SETTING_OPAQUE, valF);
 }

void MypaintView::UpdateBrushColor ()
{
  qreal h, s, v, a;
  m_color.getHsvF (&h, &s, &v, &a);
  mypaint_brush_set_base_value(mp_brush, MYPAINT_BRUSH_SETTING_COLOR_H, h);
  mypaint_brush_set_base_value(mp_brush, MYPAINT_BRUSH_SETTING_COLOR_S, s);
  mypaint_brush_set_base_value(mp_brush, MYPAINT_BRUSH_SETTING_COLOR_V, v);
}

#if defined Q_WS_WIN
  #include <windows.h>
#else
  #include <sys/time.h> // Mac & Linux
#endif

float MypaintView::GetDTime()
{
  // MyPaint is expecting :
  // Floating-point number of seconds since the last call to this,
  // function, for motion interpolation etc.
  float now; // current time is different according OS. But we just care about delta:
#if defined Q_WS_WIN
  now = (float)::GetTickCount() / 1000;
#else
  struct timeval val;
  gettimeofday(&val, NULL);
  now = (float)val.tv_sec+((float)val.tv_usec/1000000);
#endif
  float delta = now - m_lastTimeEvent;
  m_lastTimeEvent = now;
  return delta;
}
