/* brushlib - The MyPaint Brush Library (demonstration project)
 * Copyright (C) 2013 POINTCARRE SARL / Sebastien Leon email: sleon at pointcarre.com
 *
 * Permission to use, copy, modify, and/or distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 */
 
#ifndef MYPAINTVIEW_H
#define MYPAINTVIEW_H

#include <QWidget>
#include <QImage>
#include <QPoint>
#include <QTabletEvent>
#include <QGraphicsScene>
#include <QGraphicsView>
#include <QMainWindow>
#include <QDockWidget>
#include <QPushButton>
#include <QLayout>
#include <QLabel>
#include <QSlider>
#include "PuMPBrushes.h"

#undef TRUE   // redeclared in mypaint !
#undef FALSE
#include "mypaint-surface.h"
#include "mypaint-tiled-surface.h"
#include "mypaint-brush.h"
#include "mypaint-config.h"


class Tile;
// The drawing surface : will display the result of mouse/tablet press events
// and contains the current brush & the tiles in a basic 2D table...
class MypaintView : public QGraphicsView
{
  Q_OBJECT

public:
  MypaintView(QWidget* p_parent): QGraphicsView(p_parent), mp_brush(NULL), m_mousePressed(false) { Init(); }
  virtual ~MypaintView();
  enum { k_center = 50, k_max = 2*k_center, k_tile_dim = MYPAINT_TILE_SIZE };

protected:
  virtual void tabletEvent       ( QTabletEvent *event );
  virtual void mouseMoveEvent    ( QMouseEvent * event );
  virtual void mousePressEvent   ( QMouseEvent * event );
  virtual void mouseReleaseEvent ( QMouseEvent * event );
          void UpdateBrushColor  ();

private:
  void  Init();
  float GetDTime(); // return deltat Time since m_lastTimeEvent and update it to current time

  QGraphicsScene      m_scene;
  float               m_lastTimeEvent;
  MyPaintBrush*       mp_brush;
  MyPaintTiledSurface m_tile;
  QColor              m_color;
  bool                m_mousePressed;

public:
  // TILE MANAGEMENT :
  // very basic tile engine. Each tile is 128x128 pixels (biggest image is 128000 x 128000 pxls)
  // Only the requested tile are created
  Tile*          GetTileFromPos (const QPoint& pos) { return GetTileFromIdx(GetTileIndex( pos )); }
  Tile*          GetTileFromIdx (const QPoint& idx);
  inline bool    CheckIndex     (uint n) { return (n<k_max); }
  inline QPoint  GetTilePos     (const QPoint& idx) { return QPoint(k_tile_dim*idx.x(), k_tile_dim*idx.y()); }
  inline QPoint  GetTileIndex   (const QPoint& pos) { return QPoint(pos.x()/k_tile_dim, pos.y()/k_tile_dim); }
  inline QPointF GetTileFIndex  (const QPoint& pos) { return QPointF((qreal)pos.x()/k_tile_dim, (qreal)pos.y()/k_tile_dim); }
protected:
  Tile* m_tileTable [k_center*2][k_center*2]; // each allocated image is shown using the graphics view
signals:
  void BrushChanged(MyPaintBrush*);
public slots:
  void BrushSelected      (const QByteArray& content);
  void BtnChgColorPressed ();
  void RadiusChanged      (int);
  void OpacityChanged     (int);
};

// Very few things to do in this class... Just make the glue between
// the brush library widget & the drawing surface...
class MyPaintWindow : public QMainWindow
{
  Q_OBJECT
public:
  MyPaintWindow():QMainWindow(), mp_view(NULL), mp_brushes(NULL), mp_colorBtn(NULL), mp_sliderRadius(NULL), mp_sliderOpacity(NULL)
  {
    // Central widget:
    mp_view = new MypaintView(NULL);
    setCentralWidget (mp_view);

    // Add a color selector (docked):
    QDockWidget* p_dockColor = new QDockWidget("Color selection");
    mp_colorBtn = new QPushButton("Click to select another brush color...");
    mp_colorBtn->setMinimumHeight(60);
    mp_colorBtn->setStyleSheet( "color: white; background-color: black;" );
    p_dockColor->setWidget (mp_colorBtn);
    addDockWidget ( Qt::RightDockWidgetArea, p_dockColor );

    // Add a color selector (docked):
    QDockWidget* p_dockSettings = new QDockWidget("basic brush settings");
    QWidget* p_owner = new QWidget();
    QGridLayout* p_lyt = new QGridLayout(p_owner);
    p_lyt->addWidget( new QLabel("Radius:" , p_owner), 0, 0 );
    p_lyt->addWidget( new QLabel("Opacity:", p_owner), 1, 0 );
    mp_sliderRadius  = new QSlider(Qt::Horizontal, p_owner); mp_sliderRadius ->setMinimum(-200); mp_sliderRadius ->setMaximum(500);
    mp_sliderOpacity = new QSlider(Qt::Horizontal, p_owner); mp_sliderOpacity->setMinimum(   0); mp_sliderOpacity->setMaximum(200);
    p_lyt->addWidget( mp_sliderRadius , 0, 1 );
    p_lyt->addWidget( mp_sliderOpacity, 1, 1 );
    p_owner->setMaximumHeight(70);
    p_dockSettings->setWidget(p_owner);
    addDockWidget ( Qt::RightDockWidgetArea, p_dockSettings );

    // Add the brush library (docked)
    QDockWidget* p_dockBrush = new QDockWidget("Brush Library");
    mp_brushes = new PuMPBrushes( "../../brushes", NULL);
    p_dockBrush->setWidget   (mp_brushes);
    addDockWidget ( Qt::RightDockWidgetArea, p_dockBrush );
	
    connect(mp_colorBtn,      SIGNAL(pressed ()),                       mp_view, SLOT(BtnChgColorPressed()));
    connect(mp_brushes,       SIGNAL(BrushSelected(const QByteArray&)), mp_view, SLOT(BrushSelected (const QByteArray&)));
    connect(mp_sliderRadius,  SIGNAL(valueChanged(int)),                mp_view, SLOT(RadiusChanged (int)));
    connect(mp_sliderOpacity, SIGNAL(valueChanged(int)),                mp_view, SLOT(OpacityChanged(int)));
    connect(mp_view,          SIGNAL(BrushChanged(MyPaintBrush*)),               SLOT(BrushChanged(MyPaintBrush*)));
    resize(800, 700);
    mp_brushes->SelectBrush("classic/blend+paint"); // this one works fine (this is not the case with many others (why !?)
  }
protected:
  MypaintView* mp_view;
  PuMPBrushes* mp_brushes;
  QPushButton* mp_colorBtn;
  QSlider*     mp_sliderRadius;
  QSlider*     mp_sliderOpacity;
public slots:
  void BrushChanged(MyPaintBrush* p_brush)
  {
    // update basic settings according this new brush : (blocksignal avoid recursive calls)
    mp_sliderRadius ->blockSignals(true);
    mp_sliderOpacity->blockSignals(true);
    mp_sliderRadius ->setValue(100 * mypaint_brush_get_base_value(p_brush, MYPAINT_BRUSH_SETTING_RADIUS_LOGARITHMIC));
    mp_sliderOpacity->setValue(100 * mypaint_brush_get_base_value(p_brush, MYPAINT_BRUSH_SETTING_OPAQUE            ));
    mp_sliderRadius ->blockSignals(false);
    mp_sliderOpacity->blockSignals(false);
  }
};

#endif
